"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const rfc3339_1 = require("./rfc3339");
describe("RFC3339", () => {
    it("parses dates with different time zones", () => {
        // time zone +/- 0
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+00:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-00:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13)));
        // time zone positive (full hours)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+01:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 1, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+02:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 2, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+03:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 3, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+11:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 11, 12, 13)));
        // time zone negative (full hours)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-01:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 1, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-02:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 2, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-03:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 3, 12, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-11:00")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 11, 12, 13)));
        // time zone positive (minutes only)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+00:01")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 - 1, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+00:30")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 - 30, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+00:45")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 - 45, 13)));
        // time zone negative (minutes only)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-00:01")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 + 1, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-00:30")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 + 30, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-00:45")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12 + 45, 13)));
        // time zone positive (hours and minutes)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+01:01")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 1, 12 - 1, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+04:30")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 4, 12 - 30, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13+10:20")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 - 10, 12 - 20, 13)));
        // time zone negative (hours and minutes)
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-01:01")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 1, 12 + 1, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-04:30")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 4, 12 + 30, 13)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13-10:20")).toEqual(new Date(Date.UTC(2002, 9, 2, 11 + 10, 12 + 20, 13)));
    });
    it("parses dates with milliseconds", () => {
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.123Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
    });
    it("parses dates with low precision fractional seconds", () => {
        // 1 digit
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.0Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.1Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 100)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.9Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 900)));
        // 2 digit
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.00Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.12Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 120)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.99Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 990)));
    });
    it("parses dates with high precision fractional seconds", () => {
        // everything after the 3rd digit is truncated
        // 4 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.0000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.1234Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.9999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
        // 5 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.00000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.12345Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.99999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
        // 6 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.000000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.123456Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.999999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
        // 7 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.0000000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.1234567Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.9999999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
        // 8 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.00000000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.12345678Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.99999999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
        // 9 digits
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.000000000Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 0)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.123456789Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 123)));
        expect(rfc3339_1.fromRfc3339("2002-10-02T11:12:13.999999999Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 999)));
    });
    it("accepts space separators", () => {
        // https://tools.ietf.org/html/rfc3339#section-5.6
        // Applications using this syntax may choose, for the sake of readability,
        // to specify a full-date and full-time separated by (say) a space character.
        expect(rfc3339_1.fromRfc3339("2002-10-02 11:12:13Z")).toEqual(new Date(Date.UTC(2002, 9, 2, 11, 12, 13)));
    });
    it("throws for invalid format", () => {
        // extra whitespace
        expect(() => rfc3339_1.fromRfc3339(" 2002-10-02T11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13Z ")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13 Z")).toThrow();
        // wrong date separators
        expect(() => rfc3339_1.fromRfc3339("2002:10-02T11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10:02T11:12:13Z")).toThrow();
        // wrong time separators
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11-12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12-13Z")).toThrow();
        // wrong separator
        expect(() => rfc3339_1.fromRfc3339("2002-10-02TT11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02 T11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T 11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02t11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02x11:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02311:12:13Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02.11:12:13Z")).toThrow();
        // wrong time zone
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13 00:00")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2002-10-02T11:12:13+0000")).toThrow();
        // wrong fractional seconds
        expect(() => rfc3339_1.fromRfc3339("2018-07-30T19:21:12345Z")).toThrow();
        expect(() => rfc3339_1.fromRfc3339("2018-07-30T19:21:12.Z")).toThrow();
    });
    it("encodes dates", () => {
        expect(rfc3339_1.toRfc3339(new Date(Date.UTC(0, 0, 1, 0, 0, 0)))).toEqual("1900-01-01T00:00:00.000Z");
        expect(rfc3339_1.toRfc3339(new Date(Date.UTC(2002, 9, 2, 11, 12, 13, 456)))).toEqual("2002-10-02T11:12:13.456Z");
    });
});
//# sourceMappingURL=rfc3339.spec.js.map