"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const decimal_1 = require("./decimal");
const integers_1 = require("./integers");
describe("Decimal", () => {
    describe("fromAtomics", () => {
        it("leads to correct atomics value", () => {
            expect(decimal_1.Decimal.fromAtomics("1", 0).atomics).toEqual("1");
            expect(decimal_1.Decimal.fromAtomics("1", 1).atomics).toEqual("1");
            expect(decimal_1.Decimal.fromAtomics("1", 2).atomics).toEqual("1");
            expect(decimal_1.Decimal.fromAtomics("1", 5).atomics).toEqual("1");
            expect(decimal_1.Decimal.fromAtomics("2", 5).atomics).toEqual("2");
            expect(decimal_1.Decimal.fromAtomics("3", 5).atomics).toEqual("3");
            expect(decimal_1.Decimal.fromAtomics("10", 5).atomics).toEqual("10");
            expect(decimal_1.Decimal.fromAtomics("20", 5).atomics).toEqual("20");
            expect(decimal_1.Decimal.fromAtomics("30", 5).atomics).toEqual("30");
            expect(decimal_1.Decimal.fromAtomics("100000000000000000000000", 5).atomics).toEqual("100000000000000000000000");
            expect(decimal_1.Decimal.fromAtomics("200000000000000000000000", 5).atomics).toEqual("200000000000000000000000");
            expect(decimal_1.Decimal.fromAtomics("300000000000000000000000", 5).atomics).toEqual("300000000000000000000000");
            expect(decimal_1.Decimal.fromAtomics("44", 5).atomics).toEqual("44");
            expect(decimal_1.Decimal.fromAtomics("044", 5).atomics).toEqual("44");
            expect(decimal_1.Decimal.fromAtomics("0044", 5).atomics).toEqual("44");
            expect(decimal_1.Decimal.fromAtomics("00044", 5).atomics).toEqual("44");
        });
        it("reads fractional digits correctly", () => {
            expect(decimal_1.Decimal.fromAtomics("44", 0).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromAtomics("44", 1).toString()).toEqual("4.4");
            expect(decimal_1.Decimal.fromAtomics("44", 2).toString()).toEqual("0.44");
            expect(decimal_1.Decimal.fromAtomics("44", 3).toString()).toEqual("0.044");
            expect(decimal_1.Decimal.fromAtomics("44", 4).toString()).toEqual("0.0044");
        });
    });
    describe("fromUserInput", () => {
        it("throws helpful error message for invalid characters", () => {
            expect(() => decimal_1.Decimal.fromUserInput(" 13", 5)).toThrowError(/invalid character at position 1/i);
            expect(() => decimal_1.Decimal.fromUserInput("1,3", 5)).toThrowError(/invalid character at position 2/i);
            expect(() => decimal_1.Decimal.fromUserInput("13-", 5)).toThrowError(/invalid character at position 3/i);
            expect(() => decimal_1.Decimal.fromUserInput("13/", 5)).toThrowError(/invalid character at position 3/i);
            expect(() => decimal_1.Decimal.fromUserInput("13\\", 5)).toThrowError(/invalid character at position 3/i);
        });
        it("throws for more than one separator", () => {
            expect(() => decimal_1.Decimal.fromUserInput("1.3.5", 5)).toThrowError(/more than one separator found/i);
            expect(() => decimal_1.Decimal.fromUserInput("1..3", 5)).toThrowError(/more than one separator found/i);
            expect(() => decimal_1.Decimal.fromUserInput("..", 5)).toThrowError(/more than one separator found/i);
        });
        it("throws for separator only", () => {
            expect(() => decimal_1.Decimal.fromUserInput(".", 5)).toThrowError(/fractional part missing/i);
        });
        it("throws for more fractional digits than supported", () => {
            expect(() => decimal_1.Decimal.fromUserInput("44.123456", 5)).toThrowError(/got more fractional digits than supported/i);
            expect(() => decimal_1.Decimal.fromUserInput("44.1", 0)).toThrowError(/got more fractional digits than supported/i);
        });
        it("throws for fractional digits that are not non-negative integers", () => {
            // no integer
            expect(() => decimal_1.Decimal.fromUserInput("1", Number.NaN)).toThrowError(/fractional digits is not an integer/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", Number.POSITIVE_INFINITY)).toThrowError(/fractional digits is not an integer/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", Number.NEGATIVE_INFINITY)).toThrowError(/fractional digits is not an integer/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 1.78945544484)).toThrowError(/fractional digits is not an integer/i);
            // negative
            expect(() => decimal_1.Decimal.fromUserInput("1", -1)).toThrowError(/fractional digits must not be negative/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", Number.MIN_SAFE_INTEGER)).toThrowError(/fractional digits must not be negative/i);
            // exceeds supported range
            expect(() => decimal_1.Decimal.fromUserInput("1", 101)).toThrowError(/fractional digits must not exceed 100/i);
        });
        it("returns correct value", () => {
            expect(decimal_1.Decimal.fromUserInput("44", 0).atomics).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44", 1).atomics).toEqual("440");
            expect(decimal_1.Decimal.fromUserInput("44", 2).atomics).toEqual("4400");
            expect(decimal_1.Decimal.fromUserInput("44", 3).atomics).toEqual("44000");
            expect(decimal_1.Decimal.fromUserInput("44.2", 1).atomics).toEqual("442");
            expect(decimal_1.Decimal.fromUserInput("44.2", 2).atomics).toEqual("4420");
            expect(decimal_1.Decimal.fromUserInput("44.2", 3).atomics).toEqual("44200");
            expect(decimal_1.Decimal.fromUserInput("44.1", 6).atomics).toEqual("44100000");
            expect(decimal_1.Decimal.fromUserInput("44.12", 6).atomics).toEqual("44120000");
            expect(decimal_1.Decimal.fromUserInput("44.123", 6).atomics).toEqual("44123000");
            expect(decimal_1.Decimal.fromUserInput("44.1234", 6).atomics).toEqual("44123400");
            expect(decimal_1.Decimal.fromUserInput("44.12345", 6).atomics).toEqual("44123450");
            expect(decimal_1.Decimal.fromUserInput("44.123456", 6).atomics).toEqual("44123456");
        });
        it("cuts leading zeros", () => {
            expect(decimal_1.Decimal.fromUserInput("4", 2).atomics).toEqual("400");
            expect(decimal_1.Decimal.fromUserInput("04", 2).atomics).toEqual("400");
            expect(decimal_1.Decimal.fromUserInput("004", 2).atomics).toEqual("400");
        });
        it("cuts tailing zeros", () => {
            expect(decimal_1.Decimal.fromUserInput("4.12", 5).atomics).toEqual("412000");
            expect(decimal_1.Decimal.fromUserInput("4.120", 5).atomics).toEqual("412000");
            expect(decimal_1.Decimal.fromUserInput("4.1200", 5).atomics).toEqual("412000");
            expect(decimal_1.Decimal.fromUserInput("4.12000", 5).atomics).toEqual("412000");
            expect(decimal_1.Decimal.fromUserInput("4.120000", 5).atomics).toEqual("412000");
            expect(decimal_1.Decimal.fromUserInput("4.1200000", 5).atomics).toEqual("412000");
        });
        it("interprets the empty string as zero", () => {
            expect(decimal_1.Decimal.fromUserInput("", 0).atomics).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("", 1).atomics).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("", 2).atomics).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("", 3).atomics).toEqual("0");
        });
        it("accepts american notation with skipped leading zero", () => {
            expect(decimal_1.Decimal.fromUserInput(".1", 3).atomics).toEqual("100");
            expect(decimal_1.Decimal.fromUserInput(".12", 3).atomics).toEqual("120");
            expect(decimal_1.Decimal.fromUserInput(".123", 3).atomics).toEqual("123");
        });
    });
    describe("toString", () => {
        it("displays no decimal point for full numbers", () => {
            expect(decimal_1.Decimal.fromUserInput("44", 0).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44", 1).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44", 2).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44", 2).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44.0", 2).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44.00", 2).toString()).toEqual("44");
            expect(decimal_1.Decimal.fromUserInput("44.000", 2).toString()).toEqual("44");
        });
        it("only shows significant digits", () => {
            expect(decimal_1.Decimal.fromUserInput("44.1", 2).toString()).toEqual("44.1");
            expect(decimal_1.Decimal.fromUserInput("44.10", 2).toString()).toEqual("44.1");
            expect(decimal_1.Decimal.fromUserInput("44.100", 2).toString()).toEqual("44.1");
        });
        it("fills up leading zeros", () => {
            expect(decimal_1.Decimal.fromAtomics("3", 0).toString()).toEqual("3");
            expect(decimal_1.Decimal.fromAtomics("3", 1).toString()).toEqual("0.3");
            expect(decimal_1.Decimal.fromAtomics("3", 2).toString()).toEqual("0.03");
            expect(decimal_1.Decimal.fromAtomics("3", 3).toString()).toEqual("0.003");
        });
    });
    describe("toFloatApproximation", () => {
        it("works", () => {
            expect(decimal_1.Decimal.fromUserInput("0", 5).toFloatApproximation()).toEqual(0);
            expect(decimal_1.Decimal.fromUserInput("1", 5).toFloatApproximation()).toEqual(1);
            expect(decimal_1.Decimal.fromUserInput("1.5", 5).toFloatApproximation()).toEqual(1.5);
            expect(decimal_1.Decimal.fromUserInput("0.1", 5).toFloatApproximation()).toEqual(0.1);
            expect(decimal_1.Decimal.fromUserInput("1234500000000000", 5).toFloatApproximation()).toEqual(1.2345e15);
            expect(decimal_1.Decimal.fromUserInput("1234500000000000.002", 5).toFloatApproximation()).toEqual(1.2345e15);
        });
    });
    describe("plus", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.plus(decimal_1.Decimal.fromUserInput("0", 5)).toString()).toEqual("0");
            expect(zero.plus(decimal_1.Decimal.fromUserInput("1", 5)).toString()).toEqual("1");
            expect(zero.plus(decimal_1.Decimal.fromUserInput("2", 5)).toString()).toEqual("2");
            expect(zero.plus(decimal_1.Decimal.fromUserInput("2.8", 5)).toString()).toEqual("2.8");
            expect(zero.plus(decimal_1.Decimal.fromUserInput("0.12345", 5)).toString()).toEqual("0.12345");
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.plus(decimal_1.Decimal.fromUserInput("0", 5)).toString()).toEqual("1");
            expect(one.plus(decimal_1.Decimal.fromUserInput("1", 5)).toString()).toEqual("2");
            expect(one.plus(decimal_1.Decimal.fromUserInput("2", 5)).toString()).toEqual("3");
            expect(one.plus(decimal_1.Decimal.fromUserInput("2.8", 5)).toString()).toEqual("3.8");
            expect(one.plus(decimal_1.Decimal.fromUserInput("0.12345", 5)).toString()).toEqual("1.12345");
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.plus(decimal_1.Decimal.fromUserInput("0", 5)).toString()).toEqual("1.5");
            expect(oneDotFive.plus(decimal_1.Decimal.fromUserInput("1", 5)).toString()).toEqual("2.5");
            expect(oneDotFive.plus(decimal_1.Decimal.fromUserInput("2", 5)).toString()).toEqual("3.5");
            expect(oneDotFive.plus(decimal_1.Decimal.fromUserInput("2.8", 5)).toString()).toEqual("4.3");
            expect(oneDotFive.plus(decimal_1.Decimal.fromUserInput("0.12345", 5)).toString()).toEqual("1.62345");
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.plus(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
    describe("minus", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(decimal_1.Decimal.fromUserInput("0", 5).minus(zero).toString()).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("1", 5).minus(zero).toString()).toEqual("1");
            expect(decimal_1.Decimal.fromUserInput("2", 5).minus(zero).toString()).toEqual("2");
            expect(decimal_1.Decimal.fromUserInput("2.8", 5).minus(zero).toString()).toEqual("2.8");
            expect(decimal_1.Decimal.fromUserInput("0.12345", 5).minus(zero).toString()).toEqual("0.12345");
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(decimal_1.Decimal.fromUserInput("1", 5).minus(one).toString()).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("2", 5).minus(one).toString()).toEqual("1");
            expect(decimal_1.Decimal.fromUserInput("3", 5).minus(one).toString()).toEqual("2");
            expect(decimal_1.Decimal.fromUserInput("3.8", 5).minus(one).toString()).toEqual("2.8");
            expect(decimal_1.Decimal.fromUserInput("1.12345", 5).minus(one).toString()).toEqual("0.12345");
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(decimal_1.Decimal.fromUserInput("1.5", 5).minus(oneDotFive).toString()).toEqual("0");
            expect(decimal_1.Decimal.fromUserInput("2.5", 5).minus(oneDotFive).toString()).toEqual("1");
            expect(decimal_1.Decimal.fromUserInput("3.5", 5).minus(oneDotFive).toString()).toEqual("2");
            expect(decimal_1.Decimal.fromUserInput("4.3", 5).minus(oneDotFive).toString()).toEqual("2.8");
            expect(decimal_1.Decimal.fromUserInput("1.62345", 5).minus(oneDotFive).toString()).toEqual("0.12345");
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => decimal_1.Decimal.fromUserInput("1", 1).minus(zero)).toThrowError(/do not match/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 2).minus(zero)).toThrowError(/do not match/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 3).minus(zero)).toThrowError(/do not match/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 4).minus(zero)).toThrowError(/do not match/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 6).minus(zero)).toThrowError(/do not match/i);
            expect(() => decimal_1.Decimal.fromUserInput("1", 7).minus(zero)).toThrowError(/do not match/i);
        });
        it("throws for negative results", () => {
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(() => decimal_1.Decimal.fromUserInput("0", 5).minus(one)).toThrowError(/must not be negative/i);
            expect(() => decimal_1.Decimal.fromUserInput("0.5", 5).minus(one)).toThrowError(/must not be negative/i);
            expect(() => decimal_1.Decimal.fromUserInput("0.98765", 5).minus(one)).toThrowError(/must not be negative/i);
        });
    });
    describe("multiply", () => {
        it("returns correct values for Uint32", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.multiply(new integers_1.Uint32(0)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint32(1)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint32(2)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint32(4294967295)).toString()).toEqual("0");
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.multiply(new integers_1.Uint32(0)).toString()).toEqual("0");
            expect(one.multiply(new integers_1.Uint32(1)).toString()).toEqual("1");
            expect(one.multiply(new integers_1.Uint32(2)).toString()).toEqual("2");
            expect(one.multiply(new integers_1.Uint32(4294967295)).toString()).toEqual("4294967295");
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.multiply(new integers_1.Uint32(0)).toString()).toEqual("0");
            expect(oneDotFive.multiply(new integers_1.Uint32(1)).toString()).toEqual("1.5");
            expect(oneDotFive.multiply(new integers_1.Uint32(2)).toString()).toEqual("3");
            expect(oneDotFive.multiply(new integers_1.Uint32(4294967295)).toString()).toEqual("6442450942.5");
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("returns correct values for Uint53", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.multiply(new integers_1.Uint53(0)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint53(1)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint53(2)).toString()).toEqual("0");
            expect(zero.multiply(new integers_1.Uint53(9007199254740991)).toString()).toEqual("0");
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.multiply(new integers_1.Uint53(0)).toString()).toEqual("0");
            expect(one.multiply(new integers_1.Uint53(1)).toString()).toEqual("1");
            expect(one.multiply(new integers_1.Uint53(2)).toString()).toEqual("2");
            expect(one.multiply(new integers_1.Uint53(9007199254740991)).toString()).toEqual("9007199254740991");
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.multiply(new integers_1.Uint53(0)).toString()).toEqual("0");
            expect(oneDotFive.multiply(new integers_1.Uint53(1)).toString()).toEqual("1.5");
            expect(oneDotFive.multiply(new integers_1.Uint53(2)).toString()).toEqual("3");
            expect(oneDotFive.multiply(new integers_1.Uint53(9007199254740991)).toString()).toEqual("13510798882111486.5");
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("returns correct values for Uint64", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.multiply(integers_1.Uint64.fromString("0")).toString()).toEqual("0");
            expect(zero.multiply(integers_1.Uint64.fromString("1")).toString()).toEqual("0");
            expect(zero.multiply(integers_1.Uint64.fromString("2")).toString()).toEqual("0");
            expect(zero.multiply(integers_1.Uint64.fromString("18446744073709551615")).toString()).toEqual("0");
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.multiply(integers_1.Uint64.fromString("0")).toString()).toEqual("0");
            expect(one.multiply(integers_1.Uint64.fromString("1")).toString()).toEqual("1");
            expect(one.multiply(integers_1.Uint64.fromString("2")).toString()).toEqual("2");
            expect(one.multiply(integers_1.Uint64.fromString("18446744073709551615")).toString()).toEqual("18446744073709551615");
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.multiply(integers_1.Uint64.fromString("0")).toString()).toEqual("0");
            expect(oneDotFive.multiply(integers_1.Uint64.fromString("1")).toString()).toEqual("1.5");
            expect(oneDotFive.multiply(integers_1.Uint64.fromString("2")).toString()).toEqual("3");
            expect(oneDotFive.multiply(integers_1.Uint64.fromString("18446744073709551615")).toString()).toEqual("27670116110564327422.5");
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
    });
    describe("equals", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.equals(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(zero.equals(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(zero.equals(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(zero.equals(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(zero.equals(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.equals(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(one.equals(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(one.equals(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(one.equals(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(one.equals(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("1.5", 5))).toEqual(true);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(oneDotFive.equals(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.equals(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
    describe("isLessThan", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.isLessThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(zero.isLessThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(zero.isLessThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(zero.isLessThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.isLessThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(one.isLessThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(one.isLessThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(one.isLessThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(one.isLessThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("1.5", 5))).toEqual(false);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(oneDotFive.isLessThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.isLessThan(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
    describe("isLessThanOrEqual", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(one.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(one.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(one.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(one.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1.5", 5))).toEqual(true);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(true);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(true);
            expect(oneDotFive.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.isLessThanOrEqual(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
    describe("isGreaterThan", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.isGreaterThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(false);
            expect(zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(zero.isGreaterThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(zero.isGreaterThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(zero.isGreaterThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.isGreaterThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(one.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(one.isGreaterThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(one.isGreaterThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(one.isGreaterThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("1.5", 5))).toEqual(false);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(oneDotFive.isGreaterThan(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThan(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
    describe("isGreaterThanOrEqual", () => {
        it("returns correct values", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(false);
            expect(zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(false);
            const one = decimal_1.Decimal.fromUserInput("1", 5);
            expect(one.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(one.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(one.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(one.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(one.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            const oneDotFive = decimal_1.Decimal.fromUserInput("1.5", 5);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0", 5))).toEqual(true);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 5))).toEqual(true);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1.5", 5))).toEqual(true);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2", 5))).toEqual(false);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("2.8", 5))).toEqual(false);
            expect(oneDotFive.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("0.12345", 5))).toEqual(true);
            // original value remain unchanged
            expect(zero.toString()).toEqual("0");
            expect(one.toString()).toEqual("1");
            expect(oneDotFive.toString()).toEqual("1.5");
        });
        it("throws for different fractional digits", () => {
            const zero = decimal_1.Decimal.fromUserInput("0", 5);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 1))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 2))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 3))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 4))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 6))).toThrowError(/do not match/i);
            expect(() => zero.isGreaterThanOrEqual(decimal_1.Decimal.fromUserInput("1", 7))).toThrowError(/do not match/i);
        });
    });
});
//# sourceMappingURL=decimal.spec.js.map