"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyADR36Amino = exports.verifyADR36AminoSignDoc = exports.makeADR36AminoSignDoc = exports.checkAndValidateADR36AminoSignDoc = void 0;
const launchpad_1 = require("@cosmjs/launchpad");
const bech32_1 = require("../bech32");
const buffer_1 = require("buffer/");
const crypto_1 = require("@keplr-wallet/crypto");
/**
 * Check the sign doc is for ADR-36.
 * If the sign doc is expected to be ADR-36, validate the sign doc and throw an error if the sign doc is valid ADR-36.
 * @param signDoc
 * @param bech32PrefixAccAddr If this argument is provided, validate the signer in the `MsgSignData` with this prefix.
 *                            If not, validate the signer in the `MsgSignData` without considering the bech32 prefix.
 */
function checkAndValidateADR36AminoSignDoc(signDoc, bech32PrefixAccAddr) {
    const hasOnlyMsgSignData = (() => {
        if (signDoc &&
            signDoc.msgs &&
            Array.isArray(signDoc.msgs) &&
            signDoc.msgs.length === 1) {
            const msg = signDoc.msgs[0];
            return msg.type === "sign/MsgSignData";
        }
        else {
            return false;
        }
    })();
    if (!hasOnlyMsgSignData) {
        return false;
    }
    if (signDoc.chain_id !== "") {
        throw new Error("Chain id should be empty string for ADR-36 signing");
    }
    if (signDoc.memo !== "") {
        throw new Error("Memo should be empty string for ADR-36 signing");
    }
    if (signDoc.account_number !== "0") {
        throw new Error('Account number should be "0" for ADR-36 signing');
    }
    if (signDoc.sequence !== "0") {
        throw new Error('Sequence should be "0" for ADR-36 signing');
    }
    if (signDoc.fee.gas !== "0") {
        throw new Error('Gas should be "0" for ADR-36 signing');
    }
    if (signDoc.fee.amount.length !== 0) {
        throw new Error("Fee amount should be empty array for ADR-36 signing");
    }
    const msg = signDoc.msgs[0];
    if (msg.type !== "sign/MsgSignData") {
        throw new Error(`Invalid type of ADR-36 sign msg: ${msg.type}`);
    }
    if (!msg.value) {
        throw new Error("Empty value in the msg");
    }
    const signer = msg.value.signer;
    if (!signer) {
        throw new Error("Empty signer in the ADR-36 msg");
    }
    bech32_1.Bech32Address.validate(signer, bech32PrefixAccAddr);
    const data = msg.value.data;
    if (!data) {
        throw new Error("Empty data in the ADR-36 msg");
    }
    const rawData = buffer_1.Buffer.from(data, "base64");
    // Validate the data is encoded as base64.
    if (rawData.toString("base64") !== data) {
        throw new Error("Data is not encoded by base64");
    }
    if (rawData.length === 0) {
        throw new Error("Empty data in the ADR-36 msg");
    }
    return true;
}
exports.checkAndValidateADR36AminoSignDoc = checkAndValidateADR36AminoSignDoc;
function makeADR36AminoSignDoc(signer, data) {
    if (typeof data === "string") {
        data = buffer_1.Buffer.from(data).toString("base64");
    }
    else {
        data = buffer_1.Buffer.from(data).toString("base64");
    }
    return {
        chain_id: "",
        account_number: "0",
        sequence: "0",
        fee: {
            gas: "0",
            amount: [],
        },
        msgs: [
            {
                type: "sign/MsgSignData",
                value: {
                    signer,
                    data,
                },
            },
        ],
        memo: "",
    };
}
exports.makeADR36AminoSignDoc = makeADR36AminoSignDoc;
function verifyADR36AminoSignDoc(bech32PrefixAccAddr, signDoc, pubKey, signature) {
    if (!checkAndValidateADR36AminoSignDoc(signDoc, bech32PrefixAccAddr)) {
        throw new Error("Invalid sign doc for ADR-36");
    }
    const cryptoPubKey = new crypto_1.PubKeySecp256k1(pubKey);
    const expectedSigner = new bech32_1.Bech32Address(cryptoPubKey.getAddress()).toBech32(bech32PrefixAccAddr);
    const signer = signDoc.msgs[0].value.signer;
    if (expectedSigner !== signer) {
        throw new Error("Unmatched signer");
    }
    const msg = launchpad_1.serializeSignDoc(signDoc);
    return cryptoPubKey.verify(msg, signature);
}
exports.verifyADR36AminoSignDoc = verifyADR36AminoSignDoc;
function verifyADR36Amino(bech32PrefixAccAddr, signer, data, pubKey, signature) {
    const signDoc = makeADR36AminoSignDoc(signer, data);
    return verifyADR36AminoSignDoc(bech32PrefixAccAddr, signDoc, pubKey, signature);
}
exports.verifyADR36Amino = verifyADR36Amino;
//# sourceMappingURL=amino.js.map