"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CoinUtils = void 0;
const coin_1 = require("./coin");
const int_1 = require("./int");
const decimal_1 = require("./decimal");
const dec_utils_1 = require("./dec-utils");
class CoinUtils {
    static createCoinsFromPrimitives(coinPrimitives) {
        return coinPrimitives.map((primitive) => {
            return new coin_1.Coin(primitive.denom, primitive.amount);
        });
    }
    static amountOf(coins, denom) {
        const coin = coins.find((coin) => {
            return coin.denom === denom;
        });
        if (!coin) {
            return new int_1.Int(0);
        }
        else {
            return coin.amount;
        }
    }
    static exclude(coins, demons) {
        return coins.filter((coin) => {
            return demons.indexOf(coin.denom) === 0;
        });
    }
    static concat(...coins) {
        if (coins.length === 0) {
            return [];
        }
        const arr = coins.slice();
        const reducer = (accumulator, coin) => {
            // Find the duplicated denom.
            const find = accumulator.find((c) => c.denom === coin.denom);
            // If duplicated coin exists, add the amount to duplicated one.
            if (find) {
                const newCoin = new coin_1.Coin(find.denom, find.amount.add(coin.amount));
                accumulator.push(newCoin);
            }
            else {
                const newCoin = new coin_1.Coin(coin.denom, coin.amount);
                accumulator.push(newCoin);
            }
            return accumulator;
        };
        return arr.reduce(reducer, []);
    }
    static getCoinFromDecimals(currencies, decAmountStr, denom) {
        const currency = currencies.find((currency) => {
            return currency.coinDenom === denom;
        });
        if (!currency) {
            throw new Error("Invalid currency");
        }
        let precision = new decimal_1.Dec(1);
        for (let i = 0; i < currency.coinDecimals; i++) {
            precision = precision.mul(new decimal_1.Dec(10));
        }
        let decAmount = new decimal_1.Dec(decAmountStr);
        decAmount = decAmount.mul(precision);
        if (!new decimal_1.Dec(decAmount.truncate()).equals(decAmount)) {
            throw new Error("Can't divide anymore");
        }
        return new coin_1.Coin(currency.coinMinimalDenom, decAmount.truncate());
    }
    static parseDecAndDenomFromCoin(currencies, coin) {
        let currency = currencies.find((currency) => {
            return currency.coinMinimalDenom === coin.denom;
        });
        if (!currency) {
            // If the currency is unknown, just use the raw currency.
            currency = {
                coinDecimals: 0,
                coinDenom: coin.denom,
                coinMinimalDenom: coin.denom,
            };
        }
        let precision = new decimal_1.Dec(1);
        for (let i = 0; i < currency.coinDecimals; i++) {
            precision = precision.mul(new decimal_1.Dec(10));
        }
        const decAmount = new decimal_1.Dec(coin.amount).quoTruncate(precision);
        return {
            amount: decAmount.toString(currency.coinDecimals),
            denom: currency.coinDenom,
        };
    }
    static shrinkDecimals(dec, minDecimals, maxDecimals, locale = false) {
        if (dec.equals(new decimal_1.Dec(0))) {
            return "0";
        }
        const isNeg = dec.isNegative();
        const integer = dec.abs().truncate();
        const fraction = dec.abs().sub(new decimal_1.Dec(integer));
        const decimals = Math.max(maxDecimals - integer.toString().length + 1, minDecimals);
        const fractionStr = decimals === 0 ? "" : fraction.toString(decimals).replace("0.", "");
        const integerStr = locale
            ? CoinUtils.integerStringToUSLocaleString(integer.toString())
            : integer.toString();
        return ((isNeg ? "-" : "") +
            integerStr +
            (fractionStr.length > 0 ? "." : "") +
            fractionStr);
    }
    /**
     * Change the non-locale integer string to locale string.
     * Only support en-US format.
     * This method uses the BigInt if the environment supports the BigInt.
     * @param numberStr
     */
    static integerStringToUSLocaleString(numberStr) {
        if (numberStr.indexOf(".") >= 0) {
            throw new Error(`${numberStr} is not integer`);
        }
        if (typeof BigInt !== "undefined") {
            return BigInt(numberStr).toLocaleString("en-US");
        }
        const integer = numberStr;
        const chunks = [];
        for (let i = integer.length; i > 0; i -= 3) {
            chunks.push(integer.slice(Math.max(0, i - 3), i));
        }
        return chunks.reverse().join(",");
    }
    static coinToTrimmedString(coin, currency, separator = " ") {
        const dec = new decimal_1.Dec(coin.amount).quoTruncate(dec_utils_1.DecUtils.getPrecisionDec(currency.coinDecimals));
        return `${dec_utils_1.DecUtils.trim(dec)}${separator}${currency.coinDenom}`;
    }
}
exports.CoinUtils = CoinUtils;
//# sourceMappingURL=coin-utils.js.map