"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntPretty = void 0;
const int_1 = require("./int");
const decimal_1 = require("./decimal");
const dec_utils_1 = require("./dec-utils");
const coin_utils_1 = require("./coin-utils");
class IntPretty {
    constructor(num) {
        this.floatingDecimalPointRight = 0;
        this._options = {
            maxDecimals: 0,
            trim: false,
            shrink: false,
            ready: true,
            locale: true,
            inequalitySymbol: false,
            inequalitySymbolSeparator: " ",
        };
        if (typeof num === "object" && "toDec" in num) {
            num = num.toDec();
        }
        else if (!(num instanceof decimal_1.Dec)) {
            num = new decimal_1.Dec(num);
        }
        if (num.isZero()) {
            this.dec = num;
            return;
        }
        let dec = num;
        let decPrecision = 0;
        for (let i = 0; i < 18; i++) {
            if (!dec.truncate().equals(new int_1.Int(0)) &&
                dec.equals(new decimal_1.Dec(dec.truncate()))) {
                break;
            }
            dec = dec.mul(new decimal_1.Dec(10));
            decPrecision++;
        }
        this.dec = num;
        this._options.maxDecimals = decPrecision;
    }
    get options() {
        return this._options;
    }
    moveDecimalPointLeft(delta) {
        const pretty = this.clone();
        pretty.floatingDecimalPointRight += -delta;
        return pretty;
    }
    moveDecimalPointRight(delta) {
        const pretty = this.clone();
        pretty.floatingDecimalPointRight += delta;
        return pretty;
    }
    /**
     * @deprecated Use`moveDecimalPointLeft`
     */
    increasePrecision(delta) {
        return this.moveDecimalPointLeft(delta);
    }
    /**
     * @deprecated Use`moveDecimalPointRight`
     */
    decreasePrecision(delta) {
        return this.moveDecimalPointRight(delta);
    }
    maxDecimals(max) {
        const pretty = this.clone();
        pretty._options.maxDecimals = max;
        return pretty;
    }
    inequalitySymbol(bool) {
        const pretty = this.clone();
        pretty._options.inequalitySymbol = bool;
        return pretty;
    }
    inequalitySymbolSeparator(str) {
        const pretty = this.clone();
        pretty._options.inequalitySymbolSeparator = str;
        return pretty;
    }
    trim(bool) {
        const pretty = this.clone();
        pretty._options.trim = bool;
        return pretty;
    }
    shrink(bool) {
        const pretty = this.clone();
        pretty._options.shrink = bool;
        return pretty;
    }
    locale(locale) {
        const pretty = this.clone();
        pretty._options.locale = locale;
        return pretty;
    }
    /**
     * Ready indicates the actual value is ready to show the users.
     * Even if the ready option is false, it expects that the value can be shown to users (probably as 0).
     * The method that returns prettied value may return `undefined` or `null` if the value is not ready.
     * But, alternatively, it can return the 0 value that can be shown the users anyway, but indicates that the value is not ready.
     * @param bool
     */
    ready(bool) {
        const pretty = this.clone();
        pretty._options.ready = bool;
        return pretty;
    }
    get isReady() {
        return this._options.ready;
    }
    add(target) {
        if (!(target instanceof decimal_1.Dec)) {
            target = target.toDec();
        }
        const pretty = new IntPretty(this.toDec().add(target));
        pretty._options = Object.assign({}, this._options);
        return pretty;
    }
    sub(target) {
        if (!(target instanceof decimal_1.Dec)) {
            target = target.toDec();
        }
        const pretty = new IntPretty(this.toDec().sub(target));
        pretty._options = Object.assign({}, this._options);
        return pretty;
    }
    mul(target) {
        if (!(target instanceof decimal_1.Dec)) {
            target = target.toDec();
        }
        const pretty = new IntPretty(this.toDec().mul(target));
        pretty._options = Object.assign({}, this._options);
        return pretty;
    }
    quo(target) {
        if (!(target instanceof decimal_1.Dec)) {
            target = target.toDec();
        }
        const pretty = new IntPretty(this.toDec().quo(target));
        pretty._options = Object.assign({}, this._options);
        return pretty;
    }
    toDec() {
        if (this.floatingDecimalPointRight === 0) {
            return this.dec;
        }
        else if (this.floatingDecimalPointRight > 0) {
            return this.dec.mulTruncate(dec_utils_1.DecUtils.getTenExponentN(this.floatingDecimalPointRight));
        }
        else {
            // Since a decimal in Dec cannot exceed 18, it cannot be computed at once.
            let i = -this.floatingDecimalPointRight;
            let dec = this.dec;
            while (i > 0) {
                if (i >= decimal_1.Dec.precision) {
                    dec = dec.mulTruncate(dec_utils_1.DecUtils.getTenExponentN(-decimal_1.Dec.precision));
                    i -= decimal_1.Dec.precision;
                }
                else {
                    dec = dec.mulTruncate(dec_utils_1.DecUtils.getTenExponentN(-(i % decimal_1.Dec.precision)));
                    break;
                }
            }
            return dec;
        }
    }
    toString() {
        return this.toStringWithSymbols("", "");
    }
    toStringWithSymbols(prefix, suffix) {
        const dec = this.toDec();
        if (this._options.inequalitySymbol &&
            !dec.isZero() &&
            dec.abs().lt(dec_utils_1.DecUtils.getTenExponentN(-this._options.maxDecimals))) {
            const isNeg = dec.isNegative();
            return `${isNeg ? ">" : "<"}${this._options.inequalitySymbolSeparator}${isNeg ? "-" : ""}${prefix}${dec_utils_1.DecUtils.getTenExponentN(-this._options.maxDecimals).toString(this._options.maxDecimals, this._options.locale)}${suffix}`;
        }
        let result;
        if (!this._options.shrink) {
            result = dec.toString(this._options.maxDecimals, this._options.locale);
        }
        else {
            result = coin_utils_1.CoinUtils.shrinkDecimals(dec, 0, this._options.maxDecimals, this._options.locale);
        }
        if (this._options.trim) {
            result = dec_utils_1.DecUtils.trim(result);
        }
        const isNeg = result.charAt(0) === "-";
        if (isNeg) {
            result = result.slice(1);
        }
        return `${isNeg ? "-" : ""}${prefix}${result}${suffix}`;
    }
    clone() {
        const pretty = new IntPretty(this.dec);
        pretty.dec = this.dec;
        pretty.floatingDecimalPointRight = this.floatingDecimalPointRight;
        pretty._options = Object.assign({}, this._options);
        return pretty;
    }
}
exports.IntPretty = IntPretty;
//# sourceMappingURL=int-pretty.js.map