"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendSiteMetadata = void 0;
const messages_1 = __importDefault(require("./messages"));
const utils_1 = require("./utils");
/**
 * Sends site metadata over an RPC request.
 *
 * @param engine - The JSON RPC Engine to send metadata over.
 * @param log - The logging API to use.
 */
async function sendSiteMetadata(engine, log) {
    try {
        const domainMetadata = await getSiteMetadata();
        // call engine.handle directly to avoid normal RPC request handling
        engine.handle({
            jsonrpc: '2.0',
            id: 1,
            method: 'metamask_sendDomainMetadata',
            params: domainMetadata,
        }, utils_1.NOOP);
    }
    catch (error) {
        log.error({
            message: messages_1.default.errors.sendSiteMetadata(),
            originalError: error,
        });
    }
}
exports.sendSiteMetadata = sendSiteMetadata;
/**
 * Gets site metadata and returns it
 *
 */
async function getSiteMetadata() {
    return {
        name: getSiteName(window),
        icon: await getSiteIcon(window),
    };
}
/**
 * Extracts a name for the site from the DOM
 */
function getSiteName(windowObject) {
    const { document } = windowObject;
    const siteName = document.querySelector('head > meta[property="og:site_name"]');
    if (siteName) {
        return siteName.content;
    }
    const metaTitle = document.querySelector('head > meta[name="title"]');
    if (metaTitle) {
        return metaTitle.content;
    }
    if (document.title && document.title.length > 0) {
        return document.title;
    }
    return window.location.hostname;
}
/**
 * Extracts an icon for the site from the DOM
 * @returns an icon URL
 */
async function getSiteIcon(windowObject) {
    const { document } = windowObject;
    const icons = document.querySelectorAll('head > link[rel~="icon"]');
    for (const icon of icons) {
        if (icon && (await imgExists(icon.href))) {
            return icon.href;
        }
    }
    return null;
}
/**
 * Returns whether the given image URL exists
 * @param url - the url of the image
 * @returns Whether the image exists.
 */
function imgExists(url) {
    return new Promise((resolve, reject) => {
        try {
            const img = document.createElement('img');
            img.onload = () => resolve(true);
            img.onerror = () => resolve(false);
            img.src = url;
        }
        catch (e) {
            reject(e);
        }
    });
}
//# sourceMappingURL=siteMetadata.js.map