// Copyright (C) 2016 Dmitry Chestnykh
// MIT License. See LICENSE file for details.

import { ChaCha20Poly1305 } from "./chacha20poly1305";
import { encode, decode } from "@stablelib/hex";

const testVectors = [
    // From RFC7539
    {
        key: "808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9F",
        nonce: "070000004041424344454647", // concatenated "fixed-common part" and "IV"
        ad: "50515253C0C1C2C3C4C5C6C7",
        plaintext: "4C616469657320616E642047656E746C656D656E206F662074686520636C617373206F66" +
        "202739393A204966204920636F756C64206F6666657220796F75206F6E6C79206F6E6520" +
        "74697020666F7220746865206675747572652C2073756E73637265656E20776F756C6420" +
        "62652069742E",
        result: "D31A8D34648E60DB7B86AFBC53EF7EC2A4ADED51296E08FEA9E2B5A736EE62D6" +
        "3DBEA45E8CA9671282FAFB69DA92728B1A71DE0A9E060B2905D6A5B67ECD3B36" +
        "92DDBD7F2D778B8C9803AEE328091B58FAB324E4FAD675945585808B4831D7BC" +
        "3FF4DEF08E4B7A9DE576D26586CEC64B6116" +
        "1AE10B594F09E26A7E902ECBD0600691"
    },
    // From https://github.com/ctz/cifra
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164",
        plaintext: "6D657373616765",
        result: "5D9C0A9FE7D5E52824AF504FDCE6E85FC9D80C7C2A9F38"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: undefined,
        plaintext: "6D657373616765",
        result: "5D9C0A9FE7D5E5C93AA61FC3CC66A819AC96F6CE365AEE"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "",
        result: "694530BB11356A7A747EA9BCAA860165"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D",
        result: "5D8F89ED0789F37C30EB3BFBA2D24A198D"
    },
    {
            key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
            nonce: "6E6F6E63652E6E6F6E63652E",
            ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
            plaintext: "6D65",
            result: "5D9CB6441D127B771E30408A1F974A0B956A"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573",
        result: "5D9C0AA9A0059AEF600F7B006570BA5BE05421"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D657373",
        result: "5D9C0A9F34CF6163D314BB3CEA63BE4AAA59492D"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D65737361",
        result: "5D9C0A9FE734A9140C4DB224065466786CEE7475BA"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167",
        result: "5D9C0A9FE7D597A55E6765D6B57905556F8D484D26A2"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D657373616765",
        result: "5D9C0A9FE7D5E5D0C32FD8E17D1F2D1277F5CDFE7ED02F"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D",
        result: "5D9C0A9FE7D5E5DF3A37D26160B6C1F48647CA17DA695F0B"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D65",
        result: "5D9C0A9FE7D5E5DF13DC1664ACE460C2208F5A5DC4E5DD6472"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573",
        result: "5D9C0A9FE7D5E5DF1359C139BF5146E55F13AA0D088BE91D5D0E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D657373",
        result: "5D9C0A9FE7D5E5DF13599C0653AF95C3876D53AFB23D874F0623A5"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D65737361",
        result: "5D9C0A9FE7D5E5DF13599CD8F3CFFA6DC06B891976EB83F58CA9EA50"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167",
        result: "5D9C0A9FE7D5E5DF13599CD8D2AAC03A4C3F0A74B99A423C79BE60FD4D"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D21076C4FCE07331B5159CAF5B1743337A7E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107370D8A3116508EAC3569C3FE1FF92D715"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D65",
        result: "5D9C0A9FE7D5E5DF13599CD8D21073449F522873285B867612C8BA1F2A8FA12F"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573",
        result: "5D9C0A9FE7D5E5DF13599CD8D21073448503BDCA913CD39D8B210D2042EA05C15A"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D657373",
        result: "5D9C0A9FE7D5E5DF13599CD8D210734485163FDFE713BAC773C5AB3616F6E459C2D0"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D65737361",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619F4554C5573F9AF71D28339BBF312A98B"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167",
        result: "5D9C0A9FE7D5E5DF13599CD8D210734485161952FD081F2AD290C68E95DBD5AC14431CF2"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A773CEE9DD69F288E12AF8C0D44580E7E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93F462A121793B23A01FEC3D9F735F2A70"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D65",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A9359A27257B7FDE85B905516B0536E6528C2"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599BC948430FBAB784474EC0B4BFC319174A"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D657373",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5A89D071C979654AF2B2A7157EEFF3CD01"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D65737361",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE5A08AD7FDE904C3DCC78B831BEB475259"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54FD845C1C8CBC5819751A7C492349B73C9"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F572EF6BE0A3DC38ABB2B8C243C9D223454"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F571231D298C294449B1BE2A00C69CDC670E7"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D65",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B61BFB5BD4A8588726465753C824508FA"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D6573",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0E6D88783B5E19126488A8BC78E9F0104A"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C41924F71204944FE87C16C2F6549C50A3"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4FC9D4A3DAB2B33F055E055CA3652B260"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4BEC499D2764E6A5CEDAADD3B095750F5"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C43810131FDD0255FC06F2AF776BB014CF"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4684E1EC4CF1D0996FD5C0444D62347D0"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C45A8E9217D03AE2029E4B34D548EABEFA"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C43A3E5D23F581720AD0ED97E837FB00B5"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4B33799E10C5B5B9E6859F5D84A8ED377"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4DAC7C865269DF23F03EA2573E1FC6A42"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4DB14ACD3AF354206C831F19E2B9F8489"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C481161AE2823F44BF1ED248313F3B7BF4"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C43ACE5198EC3C0C7ADE20AEC577AA1BE3"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4B428A52B13EEBDA4F427388467B44D85"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C46457A561EE2910197CE1B4D4AF4AE6F0"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C45D1A0AAA7F2F0A24F760338EA6EE8081"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C454530FB3B350E1EB25CA219E55B745C9"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C47F4C50F4BB1E5B68DFD3591FB7729C69"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4D20714D88921F9E2A744161D06DFB09D"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4094B18E3134EE0F5F65115D9039F923E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C43920757A0444C1E569E1A49D1BCD9BD4"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4645E801FF75E757F604CF9698640CED5"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C401F343E91331C4AA0AEBCFB23B7E902B"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4EAF8B9A5C25B9F3C7E83831F734C5B9B"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C468F2F563DA3488D016EFE00F86DF2D5E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C4A09F1B9248C6958E66F5CFB3CCF47C6B"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C491D7E109280A96DE0E8825210DE7EFFA"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C437D94F18FB13989765287DB32083E665"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C47DF69AE381070AA774EDFEFCCEB09004"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C49BC08863939BC52736A5129F0945049D"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "6161646161646161646161646161646161646161646161646161646161",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C44BEF88996ED7179CBD5E8FEF51DB9C08"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "616164616164616164616164616164616164616164616164616164616164",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C46FBE497265F026C42E8862F8BE9BEC0E"
    },
    {
        key: "6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E6B65792E",
        nonce: "6E6F6E63652E6E6F6E63652E",
        ad: "61616461616461616461616461616461616461616461616461616461616461",
        plaintext: "6D6573736167656D6573736167656D6573736167656D6573736167656D657373616765",
        result: "5D9C0A9FE7D5E5DF13599CD8D2107344851619523A93599B5AE54F57122B0EE715D2C455C9421F09ACEA7640FBA6B31EEE94A1"
    }
];

// TODO(dchest): add more various tests.

describe("ChaCha20Poly1305", () => {
    it("should correctly seal", () => {
        testVectors.forEach(v => {
            const aead = new ChaCha20Poly1305(decode(v.key));
            const sealed = aead.seal(decode(v.nonce), decode(v.plaintext), v.ad ? decode(v.ad) : undefined);
            expect(encode(sealed)).toBe(v.result);
        });
    });

    it("should correctly open", () => {
        testVectors.forEach(v => {
            const aead = new ChaCha20Poly1305(decode(v.key));
            const plaintext = aead.open(decode(v.nonce), decode(v.result), v.ad ? decode(v.ad) : undefined);
            expect(plaintext).not.toBeNull();
            if (plaintext) {
                expect(encode(plaintext)).toBe(v.plaintext);
            }
        });
    });

    it("should not open when ciphertext is corrupted", () => {
        const v = testVectors[0];
        const sealed = decode(v.result);
        sealed[0] ^= sealed[0];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const plaintext = aead.open(decode(v.nonce), sealed, v.ad ? decode(v.ad) : undefined);
        expect(plaintext).toBeNull();
    });

    it("should not open when tag is corrupted", () => {
        const v = testVectors[0];
        const sealed = decode(v.result);
        sealed[sealed.length - 1] ^= sealed[sealed.length - 1];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const plaintext = aead.open(decode(v.nonce), sealed, v.ad ? decode(v.ad) : undefined);
        expect(plaintext).toBeNull();
    });

    it("should seal to dst it is provided", () => {
        const v = testVectors[0];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const plaintext = decode(v.plaintext);
        const ad = v.ad ? decode(v.ad) : undefined;
        const dst = new Uint8Array(plaintext.length + aead.tagLength);
        const sealed = aead.seal(decode(v.nonce), decode(v.plaintext), ad, dst);
        expect(encode(dst)).toBe(encode(sealed));
        expect(encode(sealed)).toBe(v.result);
    });

    it("should throw if seal got dst of wrong length", () => {
        const v = testVectors[0];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const plaintext = decode(v.plaintext);
        const ad = v.ad ? decode(v.ad) : undefined;
        const dst = new Uint8Array(plaintext.length + aead.tagLength - 1); // wrong length
        expect(() =>
            aead.seal(decode(v.nonce), decode(v.plaintext), ad, dst)
        ).toThrowError(/length/);
    });

    it("should open to dst it is provided", () => {
        const v = testVectors[0];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const sealed = decode(v.result);
        const ad = v.ad ? decode(v.ad) : undefined;
        const dst = new Uint8Array(sealed.length - aead.tagLength);
        const plaintext = aead.open(decode(v.nonce), decode(v.result), ad, dst);
        expect(plaintext).not.toBeNull();
        if (plaintext) {
            expect(encode(dst)).toBe(encode(plaintext));
            expect(encode(plaintext)).toBe(v.plaintext);
        }
    });

    it("should throw if open got dst of wrong length", () => {
        const v = testVectors[0];
        const aead = new ChaCha20Poly1305(decode(v.key));
        const sealed = decode(v.result);
        const ad = v.ad ? decode(v.ad) : undefined;
        const dst = new Uint8Array(sealed.length - aead.tagLength + 1); // wrong length
        expect(() =>
            aead.open(decode(v.nonce), decode(v.result), ad, dst)
        ).toThrowError(/length/);
    });
});

