"use strict";
// Source: https://github.com/pnpm/pnpm/blob/b38d711f3892a473e20e69dd32ca7e1b439efaec/fs/hard-link-dir/src/index.ts#L4
// LICENSE (MIT): https://github.com/pnpm/pnpm/blob/b38d711f3892a473e20e69dd32ca7e1b439efaec/LICENSE
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.hardLinkDir = void 0;
const path_1 = __importDefault(require("path"));
const fs_1 = require("fs");
async function hardLinkDir(src, destDirs) {
    if (destDirs.length === 0)
        return;
    // Don't try to hard link the source directory to itself
    destDirs = destDirs.filter(destDir => path_1.default.relative(destDir, src) !== '');
    const files = await fs_1.promises.readdir(src);
    await Promise.all(files.map(async (file) => {
        if (file === 'node_modules')
            return;
        const srcFile = path_1.default.join(src, file);
        if ((await fs_1.promises.lstat(srcFile)).isDirectory()) {
            const destSubdirs = await Promise.all(destDirs.map(async (destDir) => {
                const destSubdir = path_1.default.join(destDir, file);
                try {
                    await fs_1.promises.mkdir(destSubdir, { recursive: true });
                }
                catch (err) {
                    // eslint-disable-line
                    if (err.code !== 'EEXIST')
                        throw err;
                }
                return destSubdir;
            }));
            await hardLinkDir(srcFile, destSubdirs);
            return;
        }
        await Promise.all(destDirs.map(async (destDir) => {
            const destFile = path_1.default.join(destDir, file);
            try {
                await linkOrCopyFile(srcFile, destFile);
            }
            catch (err) {
                // eslint-disable-line
                if (err.code === 'ENOENT') {
                    // Ignore broken symlinks
                    return;
                }
                throw err;
            }
        }));
    }));
}
exports.hardLinkDir = hardLinkDir;
async function linkOrCopyFile(srcFile, destFile) {
    try {
        await linkOrCopy(srcFile, destFile);
    }
    catch (err) {
        // eslint-disable-line
        if (err.code === 'ENOENT') {
            await fs_1.promises.mkdir(path_1.default.dirname(destFile), { recursive: true });
            await linkOrCopy(srcFile, destFile);
            return;
        }
        if (err.code !== 'EEXIST') {
            throw err;
        }
    }
}
/*
 * This function could be optimized because we don't really need to try linking again
 * if linking failed once.
 */
async function linkOrCopy(srcFile, destFile) {
    try {
        await fs_1.promises.link(srcFile, destFile);
    }
    catch (err) {
        // eslint-disable-line
        if (err.code !== 'EXDEV')
            throw err;
        await fs_1.promises.copyFile(srcFile, destFile);
    }
}
