"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isSpawnError = exports.normalizeError = exports.errorToString = exports.isErrorLike = exports.isErrnoException = exports.isError = exports.isObject = void 0;
/**
 * A simple type guard for objects.
 *
 * @param obj - A possible object
 */
const isObject = (obj) => typeof obj === 'object' && obj !== null;
exports.isObject = isObject;
/**
 * A type guard for `try...catch` errors.
 *
 * This function is based on:
 * https://github.com/stdlib-js/assert-is-error
 */
const isError = (error) => {
    if (!(0, exports.isObject)(error))
        return false;
    // Check for `Error` objects instantiated within the current global context.
    if (error instanceof Error)
        return true;
    // Walk the prototype tree until we find a matching object.
    while (error) {
        if (Object.prototype.toString.call(error) === '[object Error]')
            return true;
        error = Object.getPrototypeOf(error);
    }
    return false;
};
exports.isError = isError;
const isErrnoException = (error) => {
    return (0, exports.isError)(error) && 'code' in error;
};
exports.isErrnoException = isErrnoException;
/**
 * A type guard for error-like objects.
 */
const isErrorLike = (error) => (0, exports.isObject)(error) && 'message' in error;
exports.isErrorLike = isErrorLike;
/**
 * Parses errors to string, useful for getting the error message in a
 * `try...catch` statement.
 */
const errorToString = (error, fallback) => {
    if ((0, exports.isError)(error) || (0, exports.isErrorLike)(error))
        return error.message;
    if (typeof error === 'string')
        return error;
    return fallback ?? 'An unknown error has ocurred.';
};
exports.errorToString = errorToString;
/**
 * Normalizes unknown errors to the Error type, useful for working with errors
 * in a `try...catch` statement.
 */
const normalizeError = (error) => {
    if ((0, exports.isError)(error))
        return error;
    const errorMessage = (0, exports.errorToString)(error);
    // Copy over additional properties if the object is error-like.
    return (0, exports.isErrorLike)(error)
        ? Object.assign(new Error(errorMessage), error)
        : new Error(errorMessage);
};
exports.normalizeError = normalizeError;
function isSpawnError(v) {
    return (0, exports.isErrnoException)(v) && 'spawnargs' in v;
}
exports.isSpawnError = isSpawnError;
//# sourceMappingURL=index.js.map