"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.copyHTMLFiles = exports.copyFunctionLibs = exports.writePrerenderConfig = exports.writeVCConfig = exports.writeHandler = void 0;
const path_1 = require("path");
const build_utils_1 = require("@vercel/build-utils");
const esbuild_1 = require("esbuild");
const fs_extra_1 = require("fs-extra");
const writeHandler = async ({ outDir, handlerFile, }) => {
    const { major } = await (0, build_utils_1.getNodeVersion)(process.cwd());
    try {
        return await (0, esbuild_1.build)({
            entryPoints: [handlerFile],
            loader: { '.ts': 'ts' },
            outfile: (0, path_1.join)(outDir, 'index.js'),
            format: 'cjs',
            target: `node${major}`,
            platform: 'node',
            bundle: true,
            minify: true,
            define: {
                'process.env.NODE_ENV': "'production'",
            },
        });
    }
    catch (e) {
        console.error('Failed to build lambda handler', e.message);
    }
};
exports.writeHandler = writeHandler;
const writeVCConfig = async ({ functionDir, handler = 'index.js', }) => {
    const { runtime } = await (0, build_utils_1.getNodeVersion)(process.cwd());
    const config = {
        runtime,
        handler,
        launcherType: 'Nodejs',
        shouldAddHelpers: true,
    };
    const configPath = (0, path_1.join)(functionDir, '.vc-config.json');
    await (0, fs_extra_1.writeJson)(configPath, config);
};
exports.writeVCConfig = writeVCConfig;
const writePrerenderConfig = (outputPath, group) => {
    const config = {
        group,
        expiration: 600, // 10 minutes TODO: make this configurable?
    };
    (0, fs_extra_1.ensureFileSync)(outputPath);
    return (0, fs_extra_1.writeFileSync)(outputPath, JSON.stringify(config));
};
exports.writePrerenderConfig = writePrerenderConfig;
async function copyFunctionLibs({ functionDir, }) {
    /* Copies the required libs for Serverless Functions from .cache to the <name>.func folder */
    await Promise.allSettled([
        {
            src: (0, path_1.join)('.cache', 'query-engine'),
            dest: (0, path_1.join)(functionDir, '.cache', 'query-engine'),
        },
        {
            src: (0, path_1.join)('.cache', 'page-ssr'),
            dest: (0, path_1.join)(functionDir, '.cache', 'page-ssr'),
        },
        {
            src: (0, path_1.join)('.cache', 'data', 'datastore'),
            dest: (0, path_1.join)(functionDir, '.cache', 'data', 'datastore'),
        },
        {
            src: (0, path_1.join)('.cache', 'caches'),
            dest: (0, path_1.join)(functionDir, '.cache', 'caches'),
        },
    ].map(({ src, dest }) => (0, fs_extra_1.copy)(src, dest)));
}
exports.copyFunctionLibs = copyFunctionLibs;
async function copyHTMLFiles({ functionDir }) {
    /* If available, copies the 404.html and 500.html files to the <name>.func/lib folder */
    for (const htmlFile of ['404', '500']) {
        if (await (0, fs_extra_1.pathExists)((0, path_1.join)('public', `${htmlFile}.html`))) {
            try {
                await (0, fs_extra_1.copyFile)((0, path_1.join)('public', `${htmlFile}.html`), (0, path_1.join)(functionDir, `${htmlFile}.html`));
            }
            catch (e) {
                console.error('Failed to copy HTML files', e.message);
                process.exit(1);
            }
        }
    }
}
exports.copyHTMLFiles = copyHTMLFiles;
//# sourceMappingURL=build.js.map