"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAPIRoutes = exports.createServerlessFunctions = void 0;
const path_1 = require("path");
const fs_extra_1 = require("fs-extra");
const symlink_1 = require("../utils/symlink");
const build_1 = require("../handlers/build");
/**
 * Gatsby SSR/DSG on Vercel is enabled through Vercel Serverless Functions.
 * This plugin creates one Serverless Function called `_ssr.func` that is used by SSR and DSG pages through symlinks.
 * DSG is enabled through prerender functions.
 */
async function createServerlessFunctions(ssrRoutes, prefix) {
    let functionName;
    let functionDir;
    const handlerFile = (0, path_1.join)(__dirname, '../handlers/templates/ssr-handler.js');
    await Promise.all(ssrRoutes.map(async (page, index) => {
        let pathName = page.path;
        // HTML renderer
        const ssrPath = (0, path_1.join)(prefix ?? '', pathName, 'index.html');
        if (index === 0) {
            // For the first page, create the SSR Serverless Function
            functionName = `${ssrPath}.func`;
            functionDir = (0, path_1.join)('.vercel/output/functions', functionName);
            await (0, fs_extra_1.ensureDir)(functionDir);
            await Promise.all([
                (0, build_1.writeHandler)({ outDir: functionDir, handlerFile }),
                (0, build_1.copyFunctionLibs)({ functionDir }),
                (0, build_1.copyHTMLFiles)({ functionDir }),
                (0, build_1.writeVCConfig)({ functionDir }),
            ]);
        }
        else {
            // If it's not the first page, then symlink to the first function
            await (0, symlink_1.createSymlink)(ssrPath, functionName);
        }
        if (page.mode === 'DSG') {
            (0, build_1.writePrerenderConfig)((0, path_1.join)('.vercel', 'output', 'functions', `${ssrPath}.prerender-config.json`), index + 1);
        }
        // page-data renderer
        if (!pathName || pathName === '/') {
            pathName = 'index';
        }
        const pageDataPath = (0, path_1.join)(prefix ?? '', 'page-data', pathName, 'page-data.json');
        await (0, symlink_1.createSymlink)(pageDataPath, functionName);
        if (page.mode === 'DSG') {
            (0, build_1.writePrerenderConfig)((0, path_1.join)('.vercel', 'output', 'functions', `${pageDataPath}.prerender-config.json`), index + 1);
        }
    }));
}
exports.createServerlessFunctions = createServerlessFunctions;
async function createAPIRoutes(functions, prefix) {
    const apiDir = (0, path_1.join)('.vercel', 'output', 'functions', 'api', prefix ?? '');
    await (0, fs_extra_1.ensureDir)(apiDir);
    await Promise.allSettled(functions.map(async (func) => {
        const apiRouteDir = `${apiDir}/${func.functionRoute}.func`;
        const handlerFile = func.originalAbsoluteFilePath;
        await (0, fs_extra_1.ensureDir)(apiRouteDir);
        await Promise.all([
            (0, build_1.writeHandler)({ outDir: apiRouteDir, handlerFile }),
            (0, build_1.writeVCConfig)({ functionDir: apiRouteDir }),
        ]);
    }));
}
exports.createAPIRoutes = createAPIRoutes;
//# sourceMappingURL=functions.js.map