"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.readMessage = exports.forkDevServer = void 0;
const once_1 = __importDefault(require("@tootallnate/once"));
const build_utils_1 = require("@vercel/build-utils");
const child_process_1 = require("child_process");
const url_1 = require("url");
const path_1 = require("path");
function forkDevServer(options) {
    let nodeOptions = process.env.NODE_OPTIONS || '';
    if (!nodeOptions.includes('--no-warnings')) {
        nodeOptions += ' --no-warnings';
    }
    const tsNodePath = options.require_.resolve('ts-node');
    const esmLoader = (0, url_1.pathToFileURL)((0, path_1.join)(tsNodePath, '..', '..', 'esm.mjs'));
    const cjsLoader = (0, path_1.join)(tsNodePath, '..', '..', 'register', 'index.js');
    const devServerPath = options.devServerPath || (0, path_1.join)(__dirname, 'dev-server.mjs');
    if (options.maybeTranspile) {
        if (options.isTypeScript) {
            nodeOptions = `--loader ${esmLoader} ${nodeOptions || ''}`;
        }
        else {
            if (options.isEsm) {
                // no transform needed because Node.js supports ESM natively
            }
            else {
                nodeOptions = `--require ${cjsLoader} ${nodeOptions || ''}`;
            }
        }
    }
    const forkOptions = {
        cwd: options.workPath,
        execArgv: [],
        env: (0, build_utils_1.cloneEnv)(process.env, options.meta.env, {
            VERCEL_DEV_ENTRYPOINT: options.entrypoint,
            VERCEL_DEV_CONFIG: JSON.stringify(options.config),
            VERCEL_DEV_BUILD_ENV: JSON.stringify(options.meta.buildEnv || {}),
            TS_NODE_TRANSPILE_ONLY: '1',
            TS_NODE_COMPILER_OPTIONS: options.tsConfig?.compilerOptions
                ? JSON.stringify(options.tsConfig.compilerOptions)
                : undefined,
            NODE_OPTIONS: nodeOptions,
        }),
    };
    const child = (0, child_process_1.fork)(devServerPath, [], forkOptions);
    checkForPid(devServerPath, child);
    return child;
}
exports.forkDevServer = forkDevServer;
function checkForPid(path, process) {
    if (!process.pid) {
        throw new Error(`Child Process has no "pid" when forking: "${path}"`);
    }
}
/**
 * When launching a dev-server, we want to know its state.
 * This function will be used to know whether it was exited (due to some error),
 * or it is listening to new requests, and we can start proxying requests.
 */
async function readMessage(child) {
    const onMessage = (0, once_1.default)(child, 'message');
    const onExit = once_1.default.spread(child, 'close');
    const result = await Promise.race([
        onMessage.then(x => {
            return { state: 'message', value: x };
        }),
        onExit.then(v => {
            return { state: 'exit', value: v };
        }),
    ]);
    onExit.cancel();
    onMessage.cancel();
    return result;
}
exports.readMessage = readMessage;
