"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getWebExportsHandler = void 0;
const node_utils_1 = require("@edge-runtime/node-utils");
class FetchEvent {
    constructor(request) {
        this.request = request;
        this.response = null;
        this.awaiting = new Set();
    }
    respondWith(response) {
        this.response = response;
    }
    waitUntil() {
        throw new Error('waitUntil is not implemented yet for Node.js');
    }
}
const webHandlerToNodeHandler = (0, node_utils_1.buildToNodeHandler)({
    Headers,
    ReadableStream,
    Request: class extends Request {
        constructor(input, init) {
            super(input, addDuplexToInit(init));
        }
    },
    Uint8Array: Uint8Array,
    FetchEvent: FetchEvent,
}, { defaultOrigin: 'https://vercel.com' });
/**
 * When users export at least one HTTP handler, we will generate
 * a generic handler routing to the right method. If there is no
 * handler function exported returns null.
 */
function getWebExportsHandler(listener, methods) {
    const handlerByMethod = {};
    for (const key of methods) {
        handlerByMethod[key] =
            typeof listener[key] !== 'undefined'
                ? webHandlerToNodeHandler(listener[key])
                : defaultHttpHandler;
    }
    return (req, res) => {
        const method = req.method ?? 'GET';
        handlerByMethod[method](req, res);
    };
}
exports.getWebExportsHandler = getWebExportsHandler;
/**
 * Add `duplex: 'half'` by default to all requests
 * https://github.com/vercel/edge-runtime/blob/bf167c418247a79d3941bfce4a5d43c37f512502/packages/primitives/src/primitives/fetch.js#L22-L26
 * https://developer.chrome.com/articles/fetch-streaming-requests/#streaming-request-bodies
 */
function addDuplexToInit(init) {
    if (typeof init === 'undefined' || typeof init === 'object') {
        return { duplex: 'half', ...init };
    }
    return init;
}
function defaultHttpHandler(_, res) {
    res.statusCode = 405;
    res.end();
}
