"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.fetchFlags = exports.getFlattenedFlagsFromChangeset = exports.camelCaseKeys = void 0;
const lodash_camelcase_1 = __importDefault(require("lodash.camelcase"));
/**
 * Transforms a set of flags so that their keys are camelCased. This function ignores
 * flag keys which start with `$`.
 *
 * @param rawFlags A mapping of flag keys and their values
 * @return A transformed `LDFlagSet` with camelCased flag keys
 */
const camelCaseKeys = (rawFlags) => {
    const flags = {};
    for (const rawFlag in rawFlags) {
        // Exclude system keys
        if (rawFlag.indexOf('$') !== 0) {
            flags[(0, lodash_camelcase_1.default)(rawFlag)] = rawFlags[rawFlag]; // tslint:disable-line:no-unsafe-any
        }
    }
    return flags;
};
exports.camelCaseKeys = camelCaseKeys;
/**
 * Gets the flags to pass to the provider from the changeset.
 *
 * @param changes the `LDFlagChangeset` from the ldClient onchange handler.
 * @param targetFlags if targetFlags are specified, changes to other flags are ignored and not returned in the
 * flattened `LDFlagSet`
 * @return an `LDFlagSet` with the current flag values from the LDFlagChangeset filtered by `targetFlags`. The returned
 * object may be empty `{}` if none of the targetFlags were changed.
 */
const getFlattenedFlagsFromChangeset = (changes, targetFlags) => {
    const flattened = {};
    for (const key in changes) {
        if (!targetFlags || targetFlags[key] !== undefined) {
            flattened[key] = changes[key].current;
        }
    }
    return flattened;
};
exports.getFlattenedFlagsFromChangeset = getFlattenedFlagsFromChangeset;
/**
 * Retrieves flag values.
 *
 * @param ldClient LaunchDarkly client
 * @param targetFlags If specified, `launchdarkly-react-client-sdk` will only request and listen to these flags.
 * Flag keys must be in their original form as known to LaunchDarkly rather than in their camel-cased form.
 *
 * @returns an `LDFlagSet` with the current flag values from LaunchDarkly filtered by `targetFlags`.
 */
const fetchFlags = (ldClient, targetFlags) => {
    const allFlags = ldClient.allFlags();
    if (!targetFlags) {
        return allFlags;
    }
    return Object.keys(targetFlags).reduce((acc, key) => {
        acc[key] = Object.prototype.hasOwnProperty.call(allFlags, key) ? allFlags[key] : targetFlags[key];
        return acc;
    }, {});
};
exports.fetchFlags = fetchFlags;
/**
 * @deprecated The `camelCaseKeys.camelCaseKeys` property will be removed in a future version,
 * please update your code to use the `camelCaseKeys` function directly.
 */
// tslint:disable-next-line deprecation
exports.camelCaseKeys.camelCaseKeys = exports.camelCaseKeys;
exports.default = { camelCaseKeys: exports.camelCaseKeys, getFlattenedFlagsFromChangeset: exports.getFlattenedFlagsFromChangeset, fetchFlags: exports.fetchFlags };
//# sourceMappingURL=utils.js.map